function [orig,term] = distmetric(varargin)
%This function calculates the distance metric.  It will use the ori
%centroids to search for all of the centroids in termi that falls within
%the search volume.
%Synatax:   [ori,termi] = distmetric('sdist',20); 
%Input:     This function does not really require any inputs.  However if
%           you do not want default values then.
%           'sdist' = the search distance.  Default = 10 pixels
%           'x' = the scale, as a ratio or xyz, of 1 unit in the x axis.
%                 Default = 1
%           'y' = the scale, as a ratio or xyz, of 1 unit in the y axis.
%                 Default = 1
%           'z' = the scale, as a ratio or xyz, of 1 unit in the z axis.
%                 Default = 1
%           'noself' = special case if you want to compare a list to
%                   itself, but don't want all points to be 0.  So, the
%                   nearest point not self.  Default = 0(off)
%           'multi' = turn on multi thread processing.  Default = 0(off)
%           'rnd' = turn on calculate random permutations. Default = 0(off)
%Output:    The structure of the data set is this:
%           origin:
%             orig.verts = An x by 3 array of vertices that was
%               used for generating the distances.
%             orig(1).ndist = an x by 1 array of the nearst distance termi
%               vertex. (2) for random verts
%             orig(1).alldist = an x by 1 array of all of the distances of
%               termi with search distance of vertex.  (2) for random verts
%             orig.filnames = the files opened
%             orig.pathnames = the place where those files were...grammer?
%           termini:
%             term(1).nverts = an x by 3 array of the vertices in termi
%               nearest to the ori vertex. (2) for random verts
%             term(1).allverts = an x by 3 array of the vertices in termi of all
%               vertices within search distance of ori vertex. (2) for random verts 
%             term.randverts = an x by 3 array of the random vertices
%               generated with similar density.
%             term.verts = an x by 3 array of termi vertices.


[sdist,x_scale,y_scale,z_scale,noself,multi,rnd] = parse(varargin);     %Parse the input

%import vertex files
%open the first ori vertex file. Prompt first
% prompt_box('title','Open Origin Vertex Lists','prompt1','Select the vertex list for the origin.','position','center');
% pause(0.25);
filenames = {};
pathnames = {};  

%get files. Note: currently the filterindex is the filter selected for the file selection.
[filename_tmp,pathname_tmp,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.xlsx','Excel 2007 files (*.xlsx)';...
    '*.xls','Excel files (*.xls)';'*.mat','Mat files (*.mat)';'*.*','All Files';},...
    'Open Origin File','Multiselect','off');
%Now open the data sets according to the filter used.
switch filterindex
    case {2,3}              %excel file
        ori = single(xlsread([pathname_tmp,filename_tmp]));
    case {1,5}                  %text file - Default coma delimited
        ori = single(dlmread([pathname_tmp,filename_tmp],',',1,0));
    otherwise               %MAT FILES      - Note Not supported right now. - why do I have it...eh!
        load([pathname_tmp,filename_tmp]);
end
filenames = cat(2,filenames,filename_tmp);
pathnames = cat(2,pathnames,pathname_tmp);
%get the ori image size
try
    ori_xyz = single(dlmread([pathname_tmp,filesep,'img_data',filesep,'img_data.csv'],',',1,0));
catch   %ok fail go to default
    ori_xyz = [max(ori(:,1)) max(ori(:,2)) max(ori(:,3))];
end
%open the second the termi vertex file. Prompt first
% prompt_box('title','Open Temini Vertex Lists','prompt1','Select the vertex list for the termini.','position','center');
% pause(0.25);
%get files. Note: currently the filterindex is the filter selected for the file selection.
[filename_tmp,pathname_tmp,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.xlsx','Excel 2007 files (*.xlsx)';...
    '*.xls','Excel files (*.xls)';'*.mat','Mat files (*.mat)';'*.*','All Files';},...
    'Open Termini File','Multiselect','off');
%Now open the data sets according to the filter used.
switch filterindex
    case {2,3}              %excel file
        termi = single(xlsread([pathname_tmp,filename_tmp]));
    case {1,5}                  %text file - Default coma delimited
        termi = single(dlmread([pathname_tmp,filename_tmp],',',1,0));
    otherwise               %MAT FILES      - Note Not supported right now. - why do I have it...eh!
        load([pathname_tmp,filename_tmp]);
end
filenames = cat(2,filenames,filename_tmp);
pathnames = cat(2,pathnames,pathname_tmp);
%get the ori image size
try
    termi_xyz = single(dlmread([pathname_tmp,filesep,'img_data',filesep,'img_data.csv'],',',1,0));
catch   %ok fail go to default
    termi_xyz = [max(termi(:,1)) max(termi(:,2)) max(termi(:,3))];
end
%clean up
clear filename_tmp pathname_tmp

%store the file properties and the vertices list
orig.filenames = filenames;
orig.pathnames = pathnames;
orig.verts = ori;
term.verts = termi;

%generate random data set.
if rnd
    % rx = randi(max(termi(:,1)),size(termi,1),1);    %random x points
    % ry = randi(max(termi(:,2)),size(termi,1),1);    %random y points
    % rz = randi(max(termi(:,3)),size(termi,1),1);    %random z points
    rx = randn(size(termi,1),1);    %random x points
    ry = randn(size(termi,1),1);    %random y points
    rz = randn(size(termi,1),1);    %random z points
    rx = (rx./max(rx)).*max(termi_xyz(1,1));     %random numbers between 0 & 1, then scatter to the volume
    ry = (ry./max(ry)).*max(termi_xyz(1,2));     %random numbers between 0 & 1, then scatter to the volume
    rz = (rz./max(rz)).*max(termi_xyz(1,3));     %random numbers between 0 & 1, then scatter to the volume
    %the random terminals will have randomized xyz, plus decimals from the
    %original data, and same size as the original termi matrix.
    if size(termi,2)>3  %there is extra info in the termi matrix
        %rtermi = horzcat((termi(:,1:3) - floor(termi(:,1:3))) + [rx ry rz],termi(:,4:end));
        rtermi = horzcat([rx ry rz],termi(:,4:end));
    else        %just vertices
        %rtermi = (termi(:,1:3) - floor(termi(:,1:3))) + [rx ry rz];
        rtermi = [rx ry rz];
    end
    %store in output structure
    term.randverts = rtermi;
    term.img_size = termi_xyz;
    %generate random data set for ori.
    rx = randn(size(ori,1),1);    %random x points
    ry = randn(size(ori,1),1);    %random y points
    rz = randn(size(ori,1),1);    %random z points
    rx = (rx./max(rx)).*max(ori_xyz(1,1));     %random numbers between 0 & 1, then scatter to the volume
    ry = (ry./max(ry)).*max(ori_xyz(1,2));     %random numbers between 0 & 1, then scatter to the volume
    rz = (rz./max(rz)).*max(ori_xyz(1,3));     %random numbers between 0 & 1, then scatter to the volume
    %the random terminals will have randomized xyz, plus decimals from the
    %original data, and same size as the original termi matrix.
    if size(ori,2)>3  %there is extra info in the termi matrix
        %rtermi = horzcat((termi(:,1:3) - floor(termi(:,1:3))) + [rx ry rz],termi(:,4:end));
        rori = horzcat([rx ry rz],ori(:,4:end));
    else        %just vertices
        %rtermi = (termi(:,1:3) - floor(termi(:,1:3))) + [rx ry rz];
        rori = [rx ry rz];
    end
    %store in output structure
    orig.randverts = rori;
    orig.img_size = ori_xyz;
end
%begin parallelization
if multi
    parpool      %initiate processes
end

%now lets calculate the distance metric
[ndist,nverts,alldist,allverts] = distance_cal(ori,termi,sdist,x_scale,y_scale,z_scale,noself,multi,'ori');
%store the data
orig.ndist = ndist;
orig.alldist = alldist;
term.nverts = nverts;
term.allverts = allverts;

if rnd
    %now lets calculate the distance metric for the random dataset
    [ndist,nverts,alldist,allverts] = distance_cal(ori,rtermi,sdist,x_scale,y_scale,z_scale,noself,multi,'random');
    %store the data
    orig(2).ndist = ndist;
    orig(2).alldist = alldist;
    term(2).nverts = nverts;
    term(2).allverts = allverts;
    
    %now lets calculate the distance metric for the random on random dataset
    [ndist,nverts,alldist,allverts] = distance_cal(rori,rtermi,sdist,x_scale,y_scale,z_scale,noself,multi,'rand on rand');
    %store the data
    orig(3).ndist = ndist;
    orig(3).alldist = alldist;
    term(3).nverts = nverts;
    term(3).allverts = allverts;
end

%closing time
if multi
    delete(gcp('nocreate'))
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [sdist,x_scale,y_scale,z_scale,noself,multi,rnd] = parse(input)

sdist = 10;  %Default Initialized.
x_scale = 1;    %base scale of a unit in the x axis
y_scale = 1;    %base scale of a unit in the y axis
z_scale = 1;    %base scale of a unit in the z axis
noself = 0;
multi = 0;
rnd = 0;

%Parse the input
if ~isempty(input)
    for i = 1:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'sdist'
                    sdist = input{1,i+1};
                case 'x'
                    x_scale = input{1,i+1};
                case 'y'
                    y_scale = input{1,i+1};
                case 'z'
                    z_scale = input{1,i+1};
                case 'noself'
                    noself = input{1,i+1};
                case 'multi'
                    multi = input{1,i+1};
                case 'rnd'
                    rnd = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end

%--------------------------------------------------------------------------
%subfunction to calculate the distance metric.
function [ndist,nverts,alldist,allverts] = distance_cal(ori,termi,sdist,x_scale,y_scale,z_scale,noself,multi,text)

%now lets calculate the distance metric
verti_tmp = ori(:,1:3).*repmat([x_scale y_scale z_scale],size(ori,1),1);   %pull data and scale the vertices to the proper unit
%     if size(ori,2)>3      %yes extra data
%         pdata_tmp = single(ori(:,4:end));      %extra data for each vertices
%     else        %no extra data
%         pdata_tmp = zeros(size(ori,1),1);      %a zero substitute
%     end
verti_co_tmp = termi(:,1:3).*repmat([x_scale y_scale z_scale],size(termi,1),1);   %scale the vertices to the proper unit
if size(termi,2)>3      %yes extra data
    pdata_co_tmp = termi(:,4:end);     %matching brightness data for comparision channel, now include other properties
else        %no extra data
    pdata_co_tmp = zeros(size(termi,1),1);      %a zero substitute
end
%preload cell arrays
ndist = zeros(size(verti_tmp,1),1);       %shortest distances
%verti_match = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching original verticies for the closeset points.
%v_match_lum = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching luminance values
%dist_verti_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %the matching distance matrix to the closest point list. A cell array as well.
%closest_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the closest pivot.  A cell array.
%closest_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the matching lumninace data.  A cell array
nverts = zeros(size(termi));       %the termi vertices
%termi_prop_tmp = zeros(size(verti_tmp,1),size(bright_tmp,2),size(filenames,2)); %the termi properties
allverts = cell(size(verti_tmp,1),1);    %The short list.  A cell Array.
%short_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list luminance.  A cell array.
alldist = cell(size(verti_tmp,1),1);     %The short list distances.  A cell Array.
%sl_verti_match_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching verticies for the short list points.
%sl_v_match_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching luminance data.
iter = size(verti_tmp,1);   %number of iterations
if ~multi
    %start progress bar
    h = waitbar(0,[text,': Vertex Distance Analyzed: 0']);    %initialize progress bar.
    for k = 1:iter     %iterate through each vertex of the set
        %sdist is modifed to increase the search area as an
        %efficent and easy way to find points in a sphere...how...wait
        [short_list short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,sdist*2,pdata_co_tmp);    %get a short list of local pivots
        curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
        if noself       %special case not self
            [y,x] = find(short_list(:,1)==curr_verti(:,1)&short_list(:,2)==curr_verti(:,2)&short_list(:,3)==curr_verti(:,3));  %find self vertex
            short_list(y,:) = [];       %gone
            short_list_lum(y,:) = [];   %gone
            curr_verti(y,:) = [];       %gone
        end
        verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
        %now we are going to make this a search in a sphere
        idx = verti_dist>sdist;     %any point actually outside of the specified radius
        verti_dist(idx) = [];       %gone
        short_list(idx,:) = [];     %gone
        short_list_lum(idx,:) = []; %gone
        if ~isempty(short_list)           %make sure there is something in the neighborhood.
            %curr_verti(idx,:) = [];     %gone
            [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
            %closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
            %closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
            if size(termi,2)>3
                nverts(k,:) = [short_list(x(1,1),:) short_list_lum(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list short_list_lum];    %output the short list as well.
            else
                nverts(k,:) = [short_list(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list];    %output the short list as well.
            end
            %verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
            %v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
            %dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
            ndist(k,1) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
            %vert_prop(k,:) = bright_tmp(k,:);         %store the vertices associated properties
            alldist{k,1} = verti_dist;  %output the distances for all points in the short list.
            %sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
            %sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
        else                        %there is no one around me (sad)
            %closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
            %closest_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
            if size(termi,2)>3
                nverts(k,:) = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];
                allverts{k,1} = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];    %output the short list as well.
            else
                nverts(k,:) = [single([NaN NaN NaN])];
                allverts{k,1} = [single([NaN NaN NaN])];    %output the short list as well.
            end
            %verti_match{k,1,j} = verti_tmp(k,:);
            %v_match_lum{k,1,j} = bright_tmp(k,:);
            %dist_verti_tmp{k,1,j} = single(NaN);
            ndist(k,1) = single(NaN);
            %vert_prop(k,:,j) = single(NaN(1,size(bright_tmp,2)));
            %short_lum_tmp{k,1,j} = single(NaN);
            alldist{k,1} = single(NaN);   %output the distance for short list.
            %sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
            %sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
        end
        waitbar(k/iter,h,[text,': Vertex Distance Analyzed: ',num2str(k)]);   %update progress
    end
    close(h);   %close progress bar
end
if multi
    parfor k = 1:iter     %iterate through each vertex of the set
        %sdist is modifed to increase the search area as an
        %efficent and easy way to find points in a sphere...how...wait
        [short_list short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,sdist*2,pdata_co_tmp);    %get a short list of local pivots
        curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
        if noself       %special case not self
            [y,x] = find(short_list(:,1)==curr_verti(:,1)&short_list(:,2)==curr_verti(:,2)&short_list(:,3)==curr_verti(:,3));  %find self vertex
            short_list(y,:) = [];       %gone
            short_list_lum(y,:) = [];   %gone
            curr_verti(y,:) = [];       %gone
        end
        verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
        %now we are going to make this a search in a sphere
        idx = verti_dist>sdist;     %any point actually outside of the specified radius
        verti_dist(idx) = [];       %gone
        short_list(idx,:) = [];     %gone
        short_list_lum(idx,:) = []; %gone
        if ~isempty(short_list)           %make sure there is something in the neighborhood.
            %curr_verti(idx,:) = [];     %gone
            [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
            %closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
            %closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
            if size(termi,2)>3
                nverts(k,:) = [short_list(x(1,1),:) short_list_lum(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list short_list_lum];    %output the short list as well.
            else
                nverts(k,:) = [short_list(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list];    %output the short list as well.
            end
            %verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
            %v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
            %dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
            ndist(k,1) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
            %vert_prop(k,:) = bright_tmp(k,:);         %store the vertices associated properties
            alldist{k,1} = verti_dist;  %output the distances for all points in the short list.
            %sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
            %sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
        else                        %there is no one around me (sad)
            %closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
            %closest_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
            if size(termi,2)>3
                nverts(k,:) = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];
                allverts{k,1} = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];    %output the short list as well.
            else
                nverts(k,:) = [single([NaN NaN NaN])];
                allverts{k,1} = [single([NaN NaN NaN])];    %output the short list as well.
            end
            %verti_match{k,1,j} = verti_tmp(k,:);
            %v_match_lum{k,1,j} = bright_tmp(k,:);
            %dist_verti_tmp{k,1,j} = single(NaN);
            ndist(k,1) = single(NaN);
            %vert_prop(k,:,j) = single(NaN(1,size(bright_tmp,2)));
            %short_lum_tmp{k,1,j} = single(NaN);
            alldist{k,1} = single(NaN);   %output the distance for short list.
            %sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
            %sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
        end
    end
end